/*
 * Decompiled with CFR 0.152.
 */
package org.apache.unomi.persistence.spi.conditions.datemath;

import java.time.DateTimeException;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.temporal.ChronoField;
import java.time.temporal.TemporalAccessor;
import java.time.temporal.TemporalQueries;
import java.util.ArrayList;
import java.util.List;
import org.apache.unomi.persistence.spi.conditions.datemath.DateMathParseException;

public class JavaDateFormatter {
    private final List<FormatDefinition> formats = new ArrayList<FormatDefinition>();
    private final boolean allowEpochMillis;
    private final boolean allowEpochSecond;

    public JavaDateFormatter(String formatString) {
        String[] formatParts;
        boolean epochMillis = false;
        boolean epochSecond = false;
        for (String f : formatParts = formatString.split("\\|\\|")) {
            if ((f = f.trim()).equals("epoch_millis")) {
                this.formats.add(new FormatDefinition("epoch_millis", f, null));
                epochMillis = true;
                continue;
            }
            if (f.equals("epoch_second")) {
                this.formats.add(new FormatDefinition("epoch_second", f, null));
                epochSecond = true;
                continue;
            }
            this.formats.add(this.createFormatDefinition(f));
        }
        this.allowEpochMillis = epochMillis;
        this.allowEpochSecond = epochSecond;
    }

    private String adjustForCaseInsensitive(String input) {
        input = input.replaceAll("(?<=\\d{4}-\\d{2}-\\d{2})t", "T");
        input = input.replaceAll("z$", "Z");
        input = input.replaceAll("(?<=[:\\d])z", "Z");
        return input;
    }

    public TemporalAccessor parse(String input) {
        if (this.isNumeric(input)) {
            long value = Long.parseLong(input);
            if (this.allowEpochMillis && this.containsFormatName("epoch_millis")) {
                return Instant.ofEpochMilli(value);
            }
            if (this.allowEpochSecond && this.containsFormatName("epoch_second")) {
                return Instant.ofEpochSecond(value);
            }
        }
        input = this.adjustForCaseInsensitive(input);
        for (FormatDefinition def : this.formats) {
            try {
                String adjusted = this.adjustForPattern(input, def.pattern);
                TemporalAccessor ta = def.formatter.parse(adjusted);
                return this.toInstant(ta, def.formatter.getZone() != null ? def.formatter.getZone() : ZoneOffset.UTC);
            }
            catch (Throwable throwable) {
            }
        }
        throw new DateMathParseException("failed to parse date field [" + input + "] with provided formats");
    }

    private Instant toInstant(TemporalAccessor ta, ZoneId zone) {
        LocalDate date;
        boolean hasYear = ta.isSupported(ChronoField.YEAR);
        boolean hasMonth = ta.isSupported(ChronoField.MONTH_OF_YEAR);
        boolean hasDay = ta.isSupported(ChronoField.DAY_OF_MONTH);
        boolean hasDoY = ta.isSupported(ChronoField.DAY_OF_YEAR);
        boolean hasHour = ta.isSupported(ChronoField.HOUR_OF_DAY);
        boolean hasMinute = ta.isSupported(ChronoField.MINUTE_OF_HOUR);
        boolean hasSecond = ta.isSupported(ChronoField.SECOND_OF_MINUTE);
        if (hasYear && hasMonth && hasDay) {
            date = LocalDate.from(ta);
        } else if (hasYear && hasDoY) {
            int year = ta.get(ChronoField.YEAR);
            int dayOfYear = ta.get(ChronoField.DAY_OF_YEAR);
            date = LocalDate.ofYearDay(year, dayOfYear);
        } else if (!hasYear && (hasHour || hasMinute || hasSecond)) {
            date = LocalDate.ofEpochDay(0L);
        } else if (hasYear && !hasMonth && !hasDay && !hasDoY) {
            int year = ta.get(ChronoField.YEAR);
            date = LocalDate.of(year, 1, 1);
        } else {
            try {
                return ZonedDateTime.from(ta).toInstant();
            }
            catch (DateTimeException e) {
                if (ta.isSupported(ChronoField.YEAR_OF_ERA)) {
                    throw new DateMathParseException("Week-based date formats need additional logic.");
                }
                date = LocalDate.ofEpochDay(0L);
            }
        }
        int hour = hasHour ? ta.get(ChronoField.HOUR_OF_DAY) : 0;
        int minute = hasMinute ? ta.get(ChronoField.MINUTE_OF_HOUR) : 0;
        int second = hasSecond ? ta.get(ChronoField.SECOND_OF_MINUTE) : 0;
        int nano = ta.isSupported(ChronoField.NANO_OF_SECOND) ? ta.get(ChronoField.NANO_OF_SECOND) : 0;
        LocalTime time = LocalTime.of(hour, minute, second, nano);
        ZoneOffset offset = ta.query(TemporalQueries.offset());
        if (offset != null) {
            return ZonedDateTime.of(date, time, offset).toInstant();
        }
        return ZonedDateTime.of(date, time, zone).toInstant();
    }

    private String adjustForPattern(String input, String pattern) {
        if (pattern.contains("strict_date") && input.matches("^\\d{4}-\\d{2}-\\d{2}$")) {
            return input + "T00:00:00Z";
        }
        return input;
    }

    private boolean isNumeric(String s) {
        if (s.isEmpty()) {
            return false;
        }
        for (char c : s.toCharArray()) {
            if (Character.isDigit(c)) continue;
            return false;
        }
        return true;
    }

    private boolean containsFormatName(String name) {
        return this.formats.stream().anyMatch(def -> def.name.equals(name));
    }

    private FormatDefinition createFormatDefinition(String f) {
        switch (f) {
            case "strict_date_optional_time": 
            case "date_optional_time": {
                return this.fmt(f, "yyyy-MM-dd['T'HH:mm:ss[.SSS][XXX]]");
            }
            case "strict_date_optional_time_nanos": {
                return this.fmt(f, "yyyy-MM-dd['T'HH:mm:ss[.SSSSSSSSS]][X]");
            }
            case "basic_date": {
                return this.fmt(f, "yyyyMMdd");
            }
            case "basic_date_time": {
                return this.fmt(f, "yyyyMMdd'T'HHmmss.SSSX");
            }
            case "basic_date_time_no_millis": {
                return this.fmt(f, "yyyyMMdd'T'HHmmssX");
            }
            case "basic_ordinal_date": {
                return this.fmt(f, "yyyyDDD");
            }
            case "basic_ordinal_date_time": {
                return this.fmt(f, "yyyyDDD'T'HHmmss.SSSX");
            }
            case "basic_ordinal_date_time_no_millis": {
                return this.fmt(f, "yyyyDDD'T'HHmmssX");
            }
            case "basic_time": {
                return this.fmt(f, "HHmmss.SSSX");
            }
            case "basic_time_no_millis": {
                return this.fmt(f, "HHmmssX");
            }
            case "basic_t_time": {
                return this.fmt(f, "'T'HHmmss.SSSX");
            }
            case "basic_t_time_no_millis": {
                return this.fmt(f, "'T'HHmmssX");
            }
            case "basic_week_date": 
            case "strict_basic_week_date": {
                return this.fmt(f, "xxxx'W'wwe");
            }
            case "basic_week_date_time": 
            case "strict_basic_week_date_time": {
                return this.fmt(f, "xxxx'W'wwe'T'HHmmss.SSSX");
            }
            case "basic_week_date_time_no_millis": 
            case "strict_basic_week_date_time_no_millis": {
                return this.fmt(f, "xxxx'W'wwe'T'HHmmssX");
            }
            case "date": 
            case "strict_date": {
                return this.fmt(f, "yyyy-MM-dd");
            }
            case "date_hour": 
            case "strict_date_hour": {
                return this.fmt(f, "yyyy-MM-dd'T'HH");
            }
            case "date_hour_minute": 
            case "strict_date_hour_minute": {
                return this.fmt(f, "yyyy-MM-dd'T'HH:mm");
            }
            case "date_hour_minute_second": 
            case "strict_date_hour_minute_second": {
                return this.fmt(f, "yyyy-MM-dd'T'HH:mm:ss");
            }
            case "date_hour_minute_second_fraction": 
            case "strict_date_hour_minute_second_fraction": {
                return this.fmt(f, "yyyy-MM-dd'T'HH:mm:ss.SSS");
            }
            case "date_hour_minute_second_millis": 
            case "strict_date_hour_minute_second_millis": {
                return this.fmt(f, "yyyy-MM-dd'T'HH:mm:ss.SSS");
            }
            case "date_time": 
            case "strict_date_time": {
                return this.fmt(f, "yyyy-MM-dd'T'HH:mm:ss.SSSX");
            }
            case "date_time_no_millis": 
            case "strict_date_time_no_millis": {
                return this.fmt(f, "yyyy-MM-dd'T'HH:mm:ssXXX");
            }
            case "hour": 
            case "strict_hour": {
                return this.fmt(f, "HH");
            }
            case "hour_minute": 
            case "strict_hour_minute": {
                return this.fmt(f, "HH:mm");
            }
            case "hour_minute_second": 
            case "strict_hour_minute_second": {
                return this.fmt(f, "HH:mm:ss");
            }
            case "hour_minute_second_fraction": 
            case "strict_hour_minute_second_fraction": {
                return this.fmt(f, "HH:mm:ss.SSS");
            }
            case "hour_minute_second_millis": 
            case "strict_hour_minute_second_millis": {
                return this.fmt(f, "HH:mm:ss.SSS");
            }
            case "ordinal_date": 
            case "strict_ordinal_date": {
                return this.fmt(f, "yyyy-DDD");
            }
            case "ordinal_date_time": 
            case "strict_ordinal_date_time": {
                return this.fmt(f, "yyyy-DDD'T'HH:mm:ss.SSSX");
            }
            case "ordinal_date_time_no_millis": 
            case "strict_ordinal_date_time_no_millis": {
                return this.fmt(f, "yyyy-DDD'T'HH:mm:ssX");
            }
            case "time": 
            case "strict_time": {
                return this.fmt(f, "HH:mm:ss.SSSX");
            }
            case "time_no_millis": 
            case "strict_time_no_millis": {
                return this.fmt(f, "HH:mm:ssX");
            }
            case "t_time": 
            case "strict_t_time": {
                return this.fmt(f, "'T'HH:mm:ss.SSSX");
            }
            case "t_time_no_millis": 
            case "strict_t_time_no_millis": {
                return this.fmt(f, "'T'HH:mm:ssX");
            }
            case "week_date": 
            case "strict_week_date": {
                return this.fmt(f, "YYYY-'W'ww-e");
            }
            case "week_date_time": 
            case "strict_week_date_time": {
                return this.fmt(f, "YYYY-'W'ww-e'T'HH:mm:ss.SSSX");
            }
            case "week_date_time_no_millis": 
            case "strict_week_date_time_no_millis": {
                return this.fmt(f, "YYYY-'W'ww-e'T'HH:mm:ssX");
            }
            case "weekyear": 
            case "strict_weekyear": {
                return this.fmt(f, "YYYY");
            }
            case "weekyear_week": 
            case "strict_weekyear_week": {
                return this.fmt(f, "YYYY-'W'ww");
            }
            case "weekyear_week_day": 
            case "strict_weekyear_week_day": {
                return this.fmt(f, "YYYY-'W'ww-e");
            }
            case "year": 
            case "strict_year": {
                return this.fmt(f, "yyyy");
            }
            case "year_month": 
            case "strict_year_month": {
                return this.fmt(f, "yyyy-MM");
            }
            case "year_month_day": 
            case "strict_year_month_day": {
                return this.fmt(f, "yyyy-MM-dd");
            }
        }
        return this.fmt(f, f);
    }

    private FormatDefinition fmt(String name, String pattern) {
        DateTimeFormatter dtf = new DateTimeFormatterBuilder().parseCaseSensitive().appendPattern(pattern).toFormatter().withZone(ZoneOffset.UTC);
        return new FormatDefinition(name, pattern, dtf);
    }

    public String toString() {
        return "JavaDateFormatter{formats=" + String.valueOf(this.formats) + "}";
    }

    public static class FormatDefinition {
        final String name;
        final String pattern;
        final DateTimeFormatter formatter;

        public FormatDefinition(String name, String pattern, DateTimeFormatter formatter) {
            this.name = name;
            this.pattern = pattern;
            this.formatter = formatter;
        }

        public String toString() {
            return "FormatDefinition{name='" + this.name + "', pattern='" + this.pattern + "'}";
        }
    }
}

