/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.netbeans.modules.php.project.connections.ui.transfer;

import java.awt.BorderLayout;
import java.util.Set;
import javax.swing.GroupLayout;
import javax.swing.GroupLayout.Alignment;
import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.LayoutStyle.ComponentPlacement;
import org.netbeans.api.annotations.common.StaticResource;
import org.netbeans.modules.php.project.connections.transfer.TransferFile;
import org.openide.awt.Mnemonics;
import org.openide.util.ImageUtilities;
import org.openide.util.NbBundle;

public final class TransferFilesChooserVisual extends JPanel {
    private static final long serialVersionUID = 8975634564231321L;

    @StaticResource
    private static final String INFO_ICON = "org/netbeans/modules/php/project/ui/resources/info_icon.png"; // NOI18N

    private final TransferFilesChooserPanel filesChooserPanel;
    private final TransferFilesChooser.TransferType transferType;

    public TransferFilesChooserVisual(TransferFilesChooserPanel filesChooserPanel, TransferFilesChooser.TransferType transferType) {
        assert filesChooserPanel != null;
        assert transferType != null;

        this.filesChooserPanel = filesChooserPanel;
        this.transferType = transferType;

        initComponents();

        innerPanel.add(filesChooserPanel.getEmbeddablePanel(), BorderLayout.CENTER);

        filesChooserPanel.addChangeListener(new TransferFilesChooserPanel.TransferFilesChangeListener() {
            @Override
            public void selectedFilesChanged() {
                updateSelectedFilesInfo();
            }

            @Override
            public void filterChanged() {
            }
        });
        updateSelectedFilesInfo();
    }

    public TransferFilesChooserPanel getEmbeddablePanel() {
        return (TransferFilesChooserPanel) outerPanel;
    }

    void updateSelectedFilesInfo() {
        String msg;
        int size = filesChooserPanel.getSelectedFiles().size();
        if (size == 0) {
            msg = NbBundle.getMessage(TransferFilesChooserVisual.class, "LBL_ZeroFilesSelected"); // NOI18N
        } else {
            // lazy download/upload
            msg = NbBundle.getMessage(TransferFilesChooserVisual.class, "LBL_FilesOrMoreSelected", size); // NOI18N
        }
        selectedFilesInfoLabel.setText(msg);
        updateWarning();
    }

    void updateWarning() {
        int size = filesChooserPanel.getSelectedFiles().size();
        if (size == 0) {
            warningLabel.setIcon(null);
            warningLabel.setText(" "); // NOI18N
        } else {
            String msgKey = null;
            switch (transferType) {
                case DOWNLOAD:
                    msgKey = "LBL_WarnDownload";
                    break;
                case UPLOAD:
                    msgKey = "LBL_WarnUpload";
                    break;
                default:
                    throw new IllegalStateException("Unknown transfer type: " + transferType);
            }
            warningLabel.setIcon(ImageUtilities.loadIcon(INFO_ICON));
            warningLabel.setText(NbBundle.getMessage(TransferFilesChooserVisual.class, msgKey));
        }
    }


    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        selectFilesLabel = new JLabel();
        outerPanel = new EmbeddablePanel(filesChooserPanel);
        innerPanel = new JPanel();
        selectedFilesInfoLabel = new JLabel();
        warningLabel = new JLabel();

        selectFilesLabel.setLabelFor(outerPanel);



        Mnemonics.setLocalizedText(selectFilesLabel, NbBundle.getMessage(TransferFilesChooserVisual.class, "TransferFilesChooserVisual.selectFilesLabel.text"));
        innerPanel.setLayout(new BorderLayout());
        Mnemonics.setLocalizedText(selectedFilesInfoLabel, "DUMMY");
        Mnemonics.setLocalizedText(warningLabel, "DUMMY");
        GroupLayout outerPanelLayout = new GroupLayout(outerPanel);
        outerPanel.setLayout(outerPanelLayout);

        outerPanelLayout.setHorizontalGroup(
            outerPanelLayout.createParallelGroup(Alignment.LEADING)
            .addGroup(outerPanelLayout.createSequentialGroup()
                .addComponent(selectedFilesInfoLabel)
                .addPreferredGap(ComponentPlacement.RELATED, 402, Short.MAX_VALUE)
                .addComponent(warningLabel))
            .addComponent(innerPanel, GroupLayout.DEFAULT_SIZE, 500, Short.MAX_VALUE)
        );
        outerPanelLayout.setVerticalGroup(
            outerPanelLayout.createParallelGroup(Alignment.LEADING)
            .addGroup(Alignment.TRAILING, outerPanelLayout.createSequentialGroup()
                .addComponent(innerPanel, GroupLayout.DEFAULT_SIZE, 305, Short.MAX_VALUE)
                .addPreferredGap(ComponentPlacement.RELATED)
                .addGroup(outerPanelLayout.createParallelGroup(Alignment.BASELINE)
                    .addComponent(selectedFilesInfoLabel)
                    .addComponent(warningLabel)))
        );

        innerPanel.getAccessibleContext().setAccessibleName(NbBundle.getMessage(TransferFilesChooserVisual.class, "TransferFilesChooserVisual.innerPanel.AccessibleContext.accessibleName")); // NOI18N
        innerPanel.getAccessibleContext().setAccessibleDescription(NbBundle.getMessage(TransferFilesChooserVisual.class, "TransferFilesChooserVisual.innerPanel.AccessibleContext.accessibleDescription")); // NOI18N
        selectedFilesInfoLabel.getAccessibleContext().setAccessibleName(NbBundle.getMessage(TransferFilesChooserVisual.class, "TransferFilesChooserVisual.selectedFilesInfoLabel.AccessibleContext.accessibleName")); // NOI18N
        selectedFilesInfoLabel.getAccessibleContext().setAccessibleDescription(NbBundle.getMessage(TransferFilesChooserVisual.class, "TransferFilesChooserVisual.selectedFilesInfoLabel.AccessibleContext.accessibleDescription")); // NOI18N
        warningLabel.getAccessibleContext().setAccessibleName(NbBundle.getMessage(TransferFilesChooserVisual.class, "TransferFilesChooserVisual.warningLabel.AccessibleContext.accessibleName")); // NOI18N
        warningLabel.getAccessibleContext().setAccessibleDescription(NbBundle.getMessage(TransferFilesChooserVisual.class, "TransferFilesChooserVisual.warningLabel.AccessibleContext.accessibleDescription")); // NOI18N
        GroupLayout layout = new GroupLayout(this);
        this.setLayout(layout);

        layout.setHorizontalGroup(
            layout.createParallelGroup(Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(Alignment.LEADING)
                    .addComponent(outerPanel, GroupLayout.DEFAULT_SIZE, GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(selectFilesLabel))
                .addContainerGap())
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(selectFilesLabel)
                .addPreferredGap(ComponentPlacement.RELATED)
                .addComponent(outerPanel, GroupLayout.DEFAULT_SIZE, GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        selectFilesLabel.getAccessibleContext().setAccessibleName(NbBundle.getMessage(TransferFilesChooserVisual.class, "TransferFilesChooserVisual.selectFilesLabel.AccessibleContext.accessibleName")); // NOI18N
        selectFilesLabel.getAccessibleContext().setAccessibleDescription(NbBundle.getMessage(TransferFilesChooserVisual.class, "TransferFilesChooserVisual.selectFilesLabel.AccessibleContext.accessibleDescription")); // NOI18N
        outerPanel.getAccessibleContext().setAccessibleName(NbBundle.getMessage(TransferFilesChooserVisual.class, "TransferFilesChooserVisual.outerPanel.AccessibleContext.accessibleName")); // NOI18N
        outerPanel.getAccessibleContext().setAccessibleDescription(NbBundle.getMessage(TransferFilesChooserVisual.class, "TransferFilesChooserVisual.outerPanel.AccessibleContext.accessibleDescription")); // NOI18N
        getAccessibleContext().setAccessibleName(NbBundle.getMessage(TransferFilesChooserVisual.class, "TransferFilesChooserVisual.AccessibleContext.accessibleName")); // NOI18N
        getAccessibleContext().setAccessibleDescription(NbBundle.getMessage(TransferFilesChooserVisual.class, "TransferFilesChooserVisual.AccessibleContext.accessibleDescription")); // NOI18N
    }// </editor-fold>//GEN-END:initComponents


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private JPanel innerPanel;
    private JPanel outerPanel;
    private JLabel selectFilesLabel;
    private JLabel selectedFilesInfoLabel;
    private JLabel warningLabel;
    // End of variables declaration//GEN-END:variables

    private static final class EmbeddablePanel extends TransferFilesChooserPanel {
        private static final long serialVersionUID = 646546111000L;

        private final TransferFilesChooserPanel delegate;

        public EmbeddablePanel(TransferFilesChooserPanel delegate) {
            assert delegate != null;

            this.delegate = delegate;
        }

        @Override
        public void addChangeListener(TransferFilesChangeListener listener) {
            delegate.addChangeListener(listener);
        }

        @Override
        public void removeChangeListener(TransferFilesChangeListener listener) {
            delegate.removeChangeListener(listener);
        }

        @Override
        public Set<TransferFile> getSelectedFiles() {
            return delegate.getSelectedFiles();
        }

        @Override
        public TransferFilesChooserPanel getEmbeddablePanel() {
            throw new IllegalStateException();
        }

        @Override
        public boolean hasAnyTransferableFiles() {
            return delegate.hasAnyTransferableFiles();
        }
    }
}
